//: Playground - noun: a place where people can play

import Cocoa

// 最小限タートルグラフィックス

// NSImageに描画する "環境" 関数を作る
extension NSImage
{
    static func imageWithBlock(_ size: NSSize, _ closure: ()->() )  -> NSImage 
    {
        let theResult = NSImage(size:size)
        
        theResult.lockFocus()
        closure()
        theResult.unlockFocus()
        
        return theResult
    }
}

// ex: 点を1つ穿つ
NSImage.imageWithBlock(NSSize(width:100, height:100))
{
    NSBezierPath(ovalIn: NSMakeRect(50, 50, 1, 1)).stroke()
}

extension CGPoint
{
    // CGPoint同士の加算を+記号で行う。
    static func + (lhs: CGPoint, rhs: CGPoint) -> CGPoint
    {
        return CGPoint(x: lhs.x + rhs.x, y: lhs.y + rhs.y)
    }
    
    // スカラーとの掛け算。lhsとrhsが型が違うので2個必要
    static func * (lhs: CGPoint, rhs: CGFloat) -> CGPoint 
    {
        return CGPoint(x: lhs.x * rhs, y: lhs.y * rhs)
    }
    
    static func * (lhs: CGFloat, rhs: CGPoint) -> CGPoint 
    {
        return CGPoint(x: lhs * rhs.x , y: lhs * rhs.y)
    }
}

struct tinyTurtle{
    let path             = NSBezierPath()
    var position         = CGPoint(x:0, y:0) // 絶対値での現在地
    var degree           = CGFloat(90.0)     // 角度。水平方向の右側が0度なので最初は上を向ける
    var penState         = true              // ペンを置いているか否か
    
    init()
    {
        self.path.move(to: position)
    }
    
    // 現在の方向から計回りに角度を曲げる
    mutating func right(_ degree: CGFloat)
    {
        self.degree -= degree
    }

    // 逆回転も追加
    mutating func left(_ degree: CGFloat)
    {
        right(-degree)
    }

    // 現在の位置から前進
    mutating func forward(_ length: CGFloat)
    {        
        // 角度と長さと現在地から、前進後の位置を求める
        let radian   = self.degree * (CGFloat.pi / 180) //ラジアンに変換
        let vector   = length * CGPoint(x: cos(radian), y: sin(radian))
        let absPoint = self.path.currentPoint + vector
        
        if penState
        {
            self.path.line(to: absPoint)
        }
        else
        {
            self.path.move(to: absPoint)
        }
    }
    
    // バックも追加
    mutating func backward(_ length: CGFloat)
    {
        forward(-length)
    }
    
    // 描画用のペンの上げ下げ
    mutating func penDown()
    {
        self.penState = true
    }

    mutating func penUp()
    {
        self.penState = false
    }

    // 絶対値で示す位置へ移動 描画ポイントが迷子になっちゃうので追加
    mutating func goto(_ x: CGFloat, _ y: CGFloat)
    {
        self.position = CGPoint(x:x, y:y)
        self.path.move(to: self.position)
    }
    
    func draw()
    {
        self.path.stroke()
    }
}

// example:四角を描く
NSImage.imageWithBlock(NSSize(width:300, height:200))
{
    var t = tinyTurtle()
    
    t.goto(150,100)
    
    t.forward(50.0)
    t.right(90)
    t.forward(50.0)
    t.right(90)
    t.forward(50.0)
    t.right(90)
    t.forward(50.0)
    t.draw()    
}

// example:四角を描く、その2
NSImage.imageWithBlock(NSSize(width:300, height:200))
{
    var t = tinyTurtle()
    
    t.goto(150,100)

    for _ in 1...4
    {
        t.forward(50.0)
        t.right(90)
    }
    t.draw()    
}
// example:三角を描く
NSImage.imageWithBlock(NSSize(width:300, height:200))
{
    var t = tinyTurtle()
    
    t.goto(150,100)
    
    for _ in 1...3
    {
        t.forward(50.0)
        t.right(120)
    }
    t.draw()    
}

// example:多角形を描く
NSImage.imageWithBlock(NSSize(width:300, height:200))
{
    var t = tinyTurtle()
    
    t.goto(150,100)
    
    func poly(_ n: Int, _ size: CGFloat)
    {
        for _ in 1...n
        {
            t.forward(size)
            t.right(360.0 / CGFloat(n))
        }        
    }
    
    poly(3, 50)
    poly(5, 50)
    poly(8, 50)
    
    t.draw()    
}
// example:円を描く
NSImage.imageWithBlock(NSSize(width:300, height:200))
{
    var t = tinyTurtle()
    
    t.goto(120,100)
    
    func poly(_ n: Int, _ size: CGFloat)
    {
        for _ in 1...n
        {
            t.forward(size)
            t.right(360.0 / CGFloat(n))
        }        
    }
    
    poly(100, 5)
    
    t.draw()    
}

// ここら辺で、末尾の"t.draw()"や先頭の"var t = tinyTurtle()"がめんどくさくなってきたのでエクステンションを書く。

extension tinyTurtle
{
    // inoutは参照渡しにするため
    static func tinyTurtleWithBlock(_ size: NSSize, _ closure: (_ t: inout tinyTurtle )->() )  -> NSImage 
    {
        let theResult = NSImage(size:size)
        
        theResult.lockFocus()
        
        var t = tinyTurtle()
        t.goto(size.width / 2, size.height / 2)

        closure(&t)// 参照渡しにするために&をつける
        
        t.draw()    

        theResult.unlockFocus()
        
        return theResult
    }
}
// こんな使い方。"t in"だけ邪魔だけど
tinyTurtle.tinyTurtleWithBlock(NSSize(width:300, height:200))
{
    t in 
    
    func poly(_ n: Int, _ size: CGFloat)
    {
        for _ in 1...n
        {
            t.forward(size)
            t.right(360.0 / CGFloat(n))
        }        
    }
    
    poly(3, 50)
    poly(5, 50)
    poly(8, 50)
}

// ドラゴン曲線を描く
tinyTurtle.tinyTurtleWithBlock(NSSize(width:300, height:200))
{
    t in 
    
    func leftDragon(_ size: CGFloat, _ n:Int)
    {
        if n == 0
        {
            t.forward(size)
            return
        }
        else
        {
            leftDragon(size, n-1)
            t.left(90)
            rightDragon(size, n-1)
        }
    }
    
    func rightDragon(_ size: CGFloat, _ n:Int)
    {
        if n == 0
        {
            t.forward(size)
            return
        }
        else
        {
            leftDragon(size, n-1)
            t.right(90)
            rightDragon(size, n-1)
        }
    }
    
    leftDragon(10, 6)// <-- ここの数字を変更すると細かさが変わる
}


// ヒルベルト曲線を描く
tinyTurtle.tinyTurtleWithBlock(NSSize(width:300, height:200))
{
    t in 
    
    func hirbert(_ size: CGFloat, _ angle:CGFloat, _ n:Int)
    {
        if n == 0
        {
            return
        }
        else
        {
            t.left(angle)
            hirbert(size, -angle, n - 1)
            t.forward(size)
            t.right(angle)
            
            hirbert(size, angle, n - 1)
            t.forward(size)
            hirbert(size, angle, n - 1)
            t.right(angle)
            t.forward(size)
            hirbert(size, -angle, n - 1)
            t.left(angle)            
        }
    }
    
    t.goto(200,20)

    hirbert(10, 90, 4)// <-- ここの数字を変更すると細かさが変わる
}

// 樹木曲線を描く。なんかあまりカッコよくない
tinyTurtle.tinyTurtleWithBlock(NSSize(width:300, height:200))
{
    t in 
    
    func tree(_ size: CGFloat, _ n: Int)
    {
        if n == 0
        {
            return
        }
        else
        {
            t.forward(size)
            t.left(50)
            tree(size/1.5, n-1)
            t.right(100)
            tree(size/1.5, n-1)
            t.left(50)
            t.backward(size)
        }
    }
    
    t.goto(150,20)

    tree(50, 6)
}

// 乱数を加えてもう少し樹木っぽくする
tinyTurtle.tinyTurtleWithBlock(NSSize(width:300, height:200))
{
    t in 
    
    func tree(_ size: CGFloat, _ n: Int)
    {
        if n == 0
        {
            return
        }
        else
        {
            t.forward(size)
            
            let r1 = CGFloat(drand48() * 60  + 15)

            t.left(r1)
            tree(size/1.5, n-1)
            t.right(r1)
            
            let r2 = CGFloat(drand48() * 60  + 15)
            
            t.right(r2)            
            tree(size/1.5, n-1)
            t.left(r2)
            
            t.backward(size)
        }
    }
    
    t.goto(150,20)
    
    tree(50, 6)
}

// コッホ曲線を描く
tinyTurtle.tinyTurtleWithBlock(NSSize(width:300, height:200))
{
    t in 
    
    func koch(_ size: CGFloat, _ angle:CGFloat, _ n:Int)
    {
        if n == 0
        {
            t.forward(size)
            return
        }
        else
        {   
            koch(size/3, angle, n-1)
            t.left(angle)
            koch(size/3, angle, n-1)            
            t.right(angle * 2)            
            koch(size/3, angle, n-1)
            t.left(angle)
            koch(size/3, angle, n-1)
        }
    }
    
    t.goto(50,20)
    t.right(90)
    
    koch(200, 60, 5)
}


